<?php
/*
 * Copyright 2018 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace Google\Auth\Tests\Cache;

use Google\Auth\Cache\SysVCacheItemPool;
use Google\Auth\Cache\TypedItem;
use Google\Auth\Tests\BaseTest;

class SysVCacheItemPoolTest extends BaseTest
{
    const VARIABLE_KEY = 99;

    private $pool;

    public function setUp(): void
    {
        if (! extension_loaded('sysvshm')) {
            $this->markTestSkipped(
                'sysvshm extension is required for running the test'
            );
        }
        $this->pool = new SysVCacheItemPool(['variableKey' => self::VARIABLE_KEY]);
        $this->pool->clear();
    }

    public function tearDown(): void
    {
        if (extension_loaded('sysvshm')) {
            $this->pool->clear();
        }
    }

    public function saveItem($key, $value)
    {
        $item = $this->pool->getItem($key);
        $item->set($value);
        $this->assertTrue($this->pool->save($item));

        return $item;
    }

    public function testGetsFreshItem()
    {
        $item = $this->pool->getItem('item');

        $this->assertInstanceOf(TypedItem::class, $item);
        $this->assertNull($item->get());
        $this->assertFalse($item->isHit());
    }

    public function testCacheAmongProcesses()
    {
        $expectedValue = 'val-' . rand();
        exec(sprintf('php %s/sysv_cache_creator.php %s', __DIR__, $expectedValue));
        $this->assertEquals(
            $expectedValue,
            $this->pool->getItem('separate-process-item')->get()
        );
    }

    public function testGetsExistingItem()
    {
        $key = 'item';
        $value = 'value';
        $this->saveItem($key, $value);
        $item = $this->pool->getItem($key);

        $this->assertInstanceOf(TypedItem::class, $item);
        $this->assertEquals($value, $item->get());
        $this->assertTrue($item->isHit());
    }

    public function testGetsMultipleItems()
    {
        $keys = ['item1', 'item2'];
        $items = $this->pool->getItems($keys);

        $this->assertEquals($keys, array_keys($items));
        $this->assertContainsOnlyInstancesOf(TypedItem::class, $items);
    }

    public function testHasItem()
    {
        $existsKey = 'does-exist';
        $this->saveItem($existsKey, 'value');

        $this->assertTrue($this->pool->hasItem($existsKey));
        $this->assertFalse($this->pool->hasItem('does-not-exist'));
    }

    public function testClear()
    {
        $key = 'item';
        $this->saveItem($key, 'value');

        $this->assertTrue($this->pool->hasItem($key));
        $this->assertTrue($this->pool->clear());
        $this->assertFalse($this->pool->hasItem($key));
    }

    public function testDeletesItem()
    {
        $key = 'item';
        $this->saveItem($key, 'value');

        $this->assertTrue($this->pool->deleteItem($key));
        $this->assertFalse($this->pool->hasItem($key));
    }

    public function testDeletesItems()
    {
        $keys = ['item1', 'item2'];

        foreach ($keys as $key) {
            $this->saveItem($key, 'value');
        }

        $this->assertTrue($this->pool->deleteItems($keys));
        $this->assertFalse($this->pool->hasItem($keys[0]));
        $this->assertFalse($this->pool->hasItem($keys[1]));
    }

    public function testSavesItem()
    {
        $key = 'item';
        $this->saveItem($key, 'value');

        $this->assertTrue($this->pool->hasItem($key));
    }

    public function testSavesDeferredItem()
    {
        $item = $this->pool->getItem('item');
        $this->assertTrue($this->pool->saveDeferred($item));
    }

    public function testCommitsDeferredItems()
    {
        $keys = ['item1', 'item2'];

        foreach ($keys as $key) {
            $item = $this->pool->getItem($key);
            $item->set('value');
            $this->pool->saveDeferred($item);
        }

        $this->assertTrue($this->pool->commit());
        $this->assertTrue($this->pool->hasItem($keys[0]));
        $this->assertTrue($this->pool->hasItem($keys[1]));
        $this->assertEquals(
            $item->get(),
            $this->pool->getItem($keys[1])->get()
        );
    }

    public function testRaceCondition()
    {
        if (!extension_loaded('sysvsem')) {
            $this->markTestSkipped(
                'sysvsem extension is required for running the race condition test'
            );
        }

        $key = 'race-item';
        $initialValue = 0;
        $this->saveItem($key, $initialValue);

        $numProcesses = 100;
        $processes = [];
        for ($i = 0; $i < $numProcesses; $i++) {
            $command = sprintf(
                'php %s/sysv_cache_race_condition_writer.php %s %s',
                __DIR__,
                $key,
                self::VARIABLE_KEY
            );
            $processes[] = proc_open($command, [], $pipes);
        }

        foreach ($processes as $process) {
            // proc_close waits for the process to terminate and returns its exit code.
            // This ensures that all child processes have completed their writes
            // before the parent process proceeds to read the final value.
            proc_close($process);
        }

        $finalValue = $this->pool->getItem($key)->get();
        $this->assertEquals($numProcesses, $finalValue);
    }
}
